<?php
/**
 * Test file to verify SQL and getCssUrl fixes
 * 
 * FIXES APPLIED:
 * 1. Fixed SQL syntax error in getFeaturedStores() - Changed from LIMIT :limit to LIMIT $limit
 * 2. Moved getCssUrl() function from header-css.php to functions.php for global availability
 * 
 * WHAT WAS WRONG:
 * - PDO cannot bind LIMIT values with named parameters in some MySQL/MariaDB versions
 * - getCssUrl() was defined in header-css.php but called before that file was included
 * 
 * NEXT STEPS IF ERRORS PERSIST:
 * 1. Make sure CDN database fields exist: run admin/sql/add_all_cdn_fields.sql
 * 2. Clear browser cache and server cache
 * 3. Check that main.js is loading (removes preloader)
 */

error_reporting(E_ALL);
ini_set('display_errors', 1);

require_once 'config.php';
require_once 'functions.php';
require_once 'core.php';

echo "<h1>Testing Fixes</h1>";
echo "<hr>";

// Test 1: Check if getCssUrl function exists
echo "<h2>Test 1: getCssUrl() Function</h2>";
if (function_exists('getCssUrl')) {
    echo "✓ getCssUrl() function exists<br>";
    try {
        $testUrl = getCssUrl('test.css');
        echo "✓ getCssUrl() works: " . htmlspecialchars($testUrl) . "<br>";
    } catch (Exception $e) {
        echo "✗ getCssUrl() error: " . $e->getMessage() . "<br>";
    }
} else {
    echo "✗ getCssUrl() function missing<br>";
}

echo "<hr>";

// Test 2: Check if getFeaturedStores works with limit
echo "<h2>Test 2: getFeaturedStores() with LIMIT</h2>";
try {
    $stores = getFeaturedStores($connect, 5);
    echo "✓ getFeaturedStores() with limit works<br>";
    echo "✓ Returned " . count($stores) . " stores<br>";
    
    if (count($stores) > 0) {
        echo "<h3>Sample Store Data:</h3>";
        $firstStore = $stores[0];
        echo "<pre>";
        echo "Store ID: " . htmlspecialchars($firstStore['store_id']) . "\n";
        echo "Store Name: " . htmlspecialchars($firstStore['store_name']) . "\n";
        echo "Store Image: " . htmlspecialchars($firstStore['store_image']) . "\n";
        echo "Total Coupons: " . htmlspecialchars($firstStore['total_items']) . "\n";
        
        // Check if CDN fields exist
        if (isset($firstStore['store_image_cdn_url'])) {
            echo "CDN URL: " . htmlspecialchars($firstStore['store_image_cdn_url']) . "\n";
            echo "Use CDN: " . htmlspecialchars($firstStore['store_use_cdn']) . "\n";
        } else {
            echo "⚠ CDN fields not in query (need to add to SELECT)\n";
        }
        echo "</pre>";
    }
} catch (Exception $e) {
    echo "✗ getFeaturedStores() error: " . $e->getMessage() . "<br>";
    echo "Error details: <pre>" . $e->getTraceAsString() . "</pre>";
}

echo "<hr>";

// Test 3: Check if getImageUrl works
echo "<h2>Test 3: getImageUrl() Function</h2>";
if (function_exists('getImageUrl')) {
    echo "✓ getImageUrl() function exists<br>";
    
    // Test with CDN enabled
    $testCdnUrl = getImageUrl('https://cdn.example.com/image.jpg', 1, 'local.jpg');
    echo "✓ CDN enabled test: " . htmlspecialchars($testCdnUrl) . "<br>";
    
    // Test with CDN disabled
    $testLocalUrl = getImageUrl('', 0, 'local.jpg');
    echo "✓ Local image test: " . htmlspecialchars($testLocalUrl) . "<br>";
} else {
    echo "✗ getImageUrl() function missing<br>";
}

echo "<hr>";
echo "<h2>Summary</h2>";
echo "<p>If all tests pass, the site should now work correctly!</p>";
echo "<p><a href='index.php'>Go to Homepage</a></p>";
?>
