<?php
// Mega Menu functionality for Slick Theme

// Add admin menu item for Mega Menu settings
add_action( 'admin_menu', 'slick_mega_menu_admin_menu' );

function slick_mega_menu_admin_menu() {
    add_submenu_page(
        'theme-options',
        __( 'Mega Menu Settings', 'slick' ),
        __( 'Mega Menu', 'slick' ),
        'manage_options',
        'slick-mega-menu',
        'slick_mega_menu_settings_page'
    );
}

// Register settings
add_action( 'admin_init', 'slick_mega_menu_settings_init' );

function slick_mega_menu_settings_init() {
    // Register settings
    register_setting( 'slick_mega_menu', 'slick_mega_menu_settings' );
    
    // Add settings section
    add_settings_section(
        'slick_mega_menu_section',
        __( 'Mega Menu Settings', 'slick' ),
        'slick_mega_menu_section_callback',
        'slick-mega-menu'
    );
    
    // Add fields
    add_settings_field(
        'mega_menu_style',
        __( 'Mega Menu Style', 'slick' ),
        'slick_mega_menu_style_callback',
        'slick-menu',
        'slick_mega_menu_section'
    );
    
    add_settings_field(
        'theme_colors',
        __( 'Theme Colors', 'slick' ),
        'slick_theme_colors_callback',
        'slick-mega-menu',
        'slick_mega_menu_section'
    );
}

// Section callback
function slick_mega_menu_section_callback() {
    echo '<p>' . __( 'Configure your mega menu settings here.', 'slick' ) . '</p>';
}

// Style callback
function slick_mega_menu_style_callback() {
    $options = get_option( 'slick_mega_menu_settings' );
    $style = isset( $options['style'] ) ? $options['style'] : 'style1';
    ?>
    <select name="slick_mega_menu_settings[style]" id="mega_menu_style">
        <option value="style1" <?php selected( $style, 'style1' ); ?>>Style 1 - Horizontal Layout</option>
        <option value="style2" <?php selected( $style, 'style2' ); ?>>Style 2 - Grid Layout</option>
        <option value="style3" <?php selected( $style, 'style3' ); ?>>Style 3 - Sidebar Layout</option>
    </select>
    <p class="description"><?php _e( 'Select the mega menu style you want to use.', 'slick' ); ?></p>
    <?php
}

// Theme colors callback
function slick_theme_colors_callback() {
    $options = get_option( 'slick_mega_menu_settings' );
    $primary_color = isset( $options['primary_color'] ) ? $options['primary_color'] : '#3498db';
    $secondary_color = isset( $options['secondary_color'] ) ? $options['secondary_color'] : '#2ecc71';
    $accent_color = isset( $options['accent_color'] ) ? $options['accent_color'] : '#e74c3c';
    ?>
    <div class="color-picker-wrapper">
        <label for="primary_color"><?php _e( 'Primary Color', 'slick' ); ?></label>
        <input type="text" name="slick_mega_menu_settings[primary_color]" class="color-picker" value="<?php echo esc_attr( $primary_color ); ?>" data-default-color="#3498db">
    </div>
    
    <div class="color-picker-wrapper">
        <label for="secondary_color"><?php _e( 'Secondary Color', 'slick' ); ?></label>
        <input type="text" name="slick_mega_menu_settings[secondary_color]" class="color-picker" value="<?php echo esc_attr( $secondary_color ); ?>" data-default-color="#2ecc71">
    </div>
    
    <div class="color-picker-wrapper">
        <label for="accent_color"><?php _e( 'Accent Color', 'slick' ); ?></label>
        <input type="text" name="slick_mega_menu_settings[accent_color]" class="color-picker" value="<?php echo esc_attr( $accent_color ); ?>" data-default-color="#e74c3c">
    </div>
    
    <style>
    .color-picker-wrapper {
        margin-bottom: 15px;
    }
    .color-picker-wrapper label {
        display: inline-block;
        width: 150px;
    }
    </style>
    <script>
    jQuery(document).ready(function($) {
        $('.color-picker').wpColorPicker();
    });
    </script>
    <?php
}

// Settings page
function slick_mega_menu_settings_page() {
    if ( ! current_user_can( 'manage_options' ) ) {
        return;
    }
    
    // Save settings if form is submitted
    if ( isset( $_POST['submit'] ) ) {
        check_admin_referer( 'slick_mega_menu_options' );
        update_option( 'slick_mega_menu_settings', $_POST['slick_mega_menu_settings'] );
        echo '<div class="notice notice-success"><p>' . __( 'Settings saved!', 'slick' ) . '</p></div>';
    }
    
    $options = get_option( 'slick_mega_menu_settings' );
    ?>
    <div class="wrap">
        <h1><?php echo esc_html( get_admin_page_title() ); ?></h1>
        <form action="" method="post">
            <?php
            settings_fields( 'slick_mega_menu' );
            do_settings_sections( 'slick-mega-menu' );
            submit_button( 'Save Settings' );
            ?>
        </form>
    </div>
    <?php
}

// Enqueue necessary scripts and styles
add_action( 'admin_enqueue_scripts', 'slick_mega_menu_admin_scripts' );

function slick_mega_menu_admin_scripts( $hook ) {
    if ( 'appearance_page_slick-mega-menu' !== $hook ) {
        return;
    }
    
    // WordPress color picker
    wp_enqueue_style( 'wp-color-picker' );
    wp_enqueue_script( 'wp-color-picker' );
    
    // Custom admin CSS
    wp_enqueue_style( 'slick-mega-menu-admin', get_template_directory_uri() . '/assets/css/mega-menu-admin.css' );
}

// Frontend styles
add_action( 'wp_enqueue_scripts', 'slick_mega_menu_frontend_scripts' );

function slick_mega_menu_frontend_scripts() {
    // Enqueue mega menu CSS
    wp_enqueue_style( 'slick-mega-menu', get_template_directory_uri() . '/assets/css/mega-menu.css' );
    
    // Enqueue mega menu JS
    wp_enqueue_script( 'slick-mega-menu', get_template_directory_uri() . '/assets/js/mega-menu.js', array( 'jquery' ), '1.0.0', true );
    
    // Get theme colors
    $options = get_option( 'slick_mega_menu_settings' );
    $primary_color = isset( $options['primary_color'] ) ? $options['primary_color'] : '#3498db';
    $secondary_color = isset( $options['secondary_color'] ) ? $options['secondary_color'] : '#2ecc71';
    $accent_color = isset( $options['accent_color'] ) ? $options['accent_color'] : '#e74c3c';
    
    // Inline styles for theme colors
    $custom_css = ":root {
        --mega-menu-primary: {$primary_color};
        --mega-menu-secondary: {$secondary_color};
        --mega-menu-accent: {$accent_color};
    }";
    
    wp_add_inline_style( 'slick-mega-menu', $custom_css );
}

// Modify menu walker to support mega menu
add_filter( 'wp_nav_menu_args', 'slick_mega_menu_walker', 1001 );

function slick_mega_menu_walker( $args ) {
    if ( 'primary' === $args['theme_location'] ) {
        if ( ! class_exists( 'Slick_Mega_Menu_Walker' ) ) {
            require_once get_template_directory() . '/inc/class-slick-mega-menu-walker.php';
        }
        $args['walker'] = new Slick_Mega_Menu_Walker();
        $args['container_class'] .= ' slick-mega-menu';
    }
    return $args;
}

// Add custom fields to menu items
add_filter( 'wp_setup_nav_menu_item', 'slick_mega_menu_custom_fields' );

function slick_mega_menu_custom_fields( $menu_item ) {
    $menu_item->mega_menu = get_post_meta( $menu_item->ID, '_menu_item_mega_menu', true );
    $menu_item->mega_menu_columns = get_post_meta( $menu_item->ID, '_menu_item_mega_menu_columns', true );
    $menu_item->mega_menu_style = get_post_meta( $menu_item->ID, '_menu_item_mega_menu_style', true );
    return $menu_item;
}

// Save custom fields
add_action( 'wp_update_nav_menu_item', 'slick_mega_menu_update', 10, 3 );

function slick_mega_menu_update( $menu_id, $menu_item_db_id, $args ) {
    if ( isset( $_REQUEST['menu-item-mega-menu'][$menu_item_db_id] ) ) {
        update_post_meta( $menu_item_db_id, '_menu_item_mega_menu', '1' );
    } else {
        delete_post_meta( $menu_item_db_id, '_menu_item_mega_menu' );
    }
    
    if ( isset( $_REQUEST['menu-item-mega-menu-columns'][$menu_item_db_id] ) ) {
        update_post_meta( $menu_item_db_id, '_menu_item_mega_menu_columns', $_REQUEST['menu-item-mega-menu-columns'][$menu_item_db_id] );
    }
    
    if ( isset( $_REQUEST['menu-item-mega-menu-style'][$menu_item_db_id] ) ) {
        update_post_meta( $menu_item_db_id, '_menu_item_mega_menu_style', $_REQUEST['menu-item-mega-menu-style'][$menu_item_db_id] );
    }
}

// Add custom fields to menu items
add_filter( 'wp_edit_nav_menu_walker', 'slick_mega_menu_edit_walker', 10, 2 );

function slick_mega_menu_edit_walker( $walker, $menu_id ) {
    require_once get_template_directory() . '/inc/class-slick-mega-menu-edit-walker.php';
    return 'Slick_Mega_Menu_Edit_Walker';
}

// Add custom CSS class to menu items
add_filter( 'nav_menu_css_class', 'slick_mega_menu_item_classes', 10, 4 );

function slick_mega_menu_item_classes( $classes, $item, $args, $depth ) {
    if ( $item->mega_menu ) {
        $classes[] = 'mega-menu';
        $classes[] = 'mega-menu-style-' . ( $item->mega_menu_style ? $item->mega_menu_style : 'default' );
    }
    return $classes;
}
?>
