<?php

// Firebase OTP Authentication Plugin
// Uses Firebase Authentication for phone number verification

if( \user\main::banned( 'login' ) || \user\main::banned( 'register' ) ) {
    header( 'Location: ' . $GLOBALS['siteURL'] );
    die;
}

// Get Firebase config from settings
$firebase_api_key = \query\main::get_option( 'firebase_api_key' );
$firebase_project_id = \query\main::get_option( 'firebase_project_id' );

if( empty( $firebase_api_key ) || empty( $firebase_project_id ) ) {
    die( 'Firebase OTP service is not configured. Please contact administrator.' );
}

// Handle OTP verification
if( isset( $_POST['action'] ) ) {
    
    $action = $_POST['action'];
    $phone = !empty( $_POST['phone'] ) ? preg_replace( '/[^0-9+]/', '', $_POST['phone'] ) : '';
    $otp = !empty( $_POST['otp'] ) ? trim( $_POST['otp'] ) : '';
    $name = !empty( $_POST['name'] ) ? trim( $_POST['name'] ) : '';
    $verification_id = !empty( $_POST['verification_id'] ) ? trim( $_POST['verification_id'] ) : '';
    
    header( 'Content-Type: application/json' );
    
    // Profile: send OTP to verify phone for logged-in user
    if( $action == 'send_profile_otp' ) {

        $me = \user\main::is_logged();
        if( !$me ) {
            echo json_encode( ['state' => 'error', 'message' => 'Please login to verify your phone number'] );
            die;
        }

        if( empty( $phone ) || strlen( $phone ) < 10 ) {
            echo json_encode( ['state' => 'error', 'message' => 'Please enter a valid phone number'] );
            die;
        }

        global $db;

        // Ensure phone columns exist
        $column_check = $db->query( "SHOW COLUMNS FROM " . DB_TABLE_PREFIX . "users LIKE 'phone'" );
        if( $column_check && $column_check->num_rows === 0 ) {
            $db->query( "ALTER TABLE " . DB_TABLE_PREFIX . "users ADD COLUMN phone VARCHAR(20) DEFAULT NULL AFTER email" );
        }
        $verified_check = $db->query( "SHOW COLUMNS FROM " . DB_TABLE_PREFIX . "users LIKE 'phone_verified'" );
        if( $verified_check && $verified_check->num_rows === 0 ) {
            $db->query( "ALTER TABLE " . DB_TABLE_PREFIX . "users ADD COLUMN phone_verified TINYINT(1) NOT NULL DEFAULT 0 AFTER phone" );
        }

        // Prevent verifying a phone that belongs to another user
        $stmt = $db->stmt_init();
        $stmt->prepare( "SELECT id FROM " . DB_TABLE_PREFIX . "users WHERE phone = ? AND id <> ? LIMIT 1" );
        $stmt->bind_param( "si", $phone, $me->ID );
        $stmt->execute();
        $stmt->bind_result( $conflict_id );
        $stmt->fetch();
        $stmt->close();
        if( !empty( $conflict_id ) ) {
            echo json_encode( ['state' => 'error', 'message' => 'This mobile number is already used by another account'] );
            die;
        }

        $verification_code = rand( 100000, 999999 );
        $_SESSION['otp_profile_verification'][$me->ID] = [
            'phone' => $phone,
            'code' => $verification_code,
            'expires' => time() + 300,
            'attempts' => 0
        ];

        echo json_encode( [
            'state' => 'success',
            'message' => 'OTP sent to ' . $phone,
            'verification_id' => md5( $me->ID . $phone . $verification_code . time() ),
            'debug_code' => $verification_code // Remove in production!
        ] );
        die;

    }

    // Profile: verify OTP for logged-in user
    if( $action == 'verify_profile_otp' ) {

        $me = \user\main::is_logged();
        if( !$me ) {
            echo json_encode( ['state' => 'error', 'message' => 'Please login to verify your phone number'] );
            die;
        }

        if( empty( $phone ) || empty( $otp ) ) {
            echo json_encode( ['state' => 'error', 'message' => 'Please enter phone number and OTP'] );
            die;
        }

        if( empty( $_SESSION['otp_profile_verification'][$me->ID] ) ) {
            echo json_encode( ['state' => 'error', 'message' => 'OTP session expired. Please request a new OTP'] );
            die;
        }

        $otp_data = $_SESSION['otp_profile_verification'][$me->ID];

        if( $otp_data['phone'] !== $phone ) {
            echo json_encode( ['state' => 'error', 'message' => 'Phone number changed. Please request a new OTP'] );
            die;
        }

        if( time() > $otp_data['expires'] ) {
            unset( $_SESSION['otp_profile_verification'][$me->ID] );
            echo json_encode( ['state' => 'error', 'message' => 'OTP expired. Please request a new OTP'] );
            die;
        }

        if( $otp_data['attempts'] >= 5 ) {
            unset( $_SESSION['otp_profile_verification'][$me->ID] );
            echo json_encode( ['state' => 'error', 'message' => 'Too many attempts. Please request a new OTP'] );
            die;
        }

        $_SESSION['otp_profile_verification'][$me->ID]['attempts']++;

        if( (string) $otp !== (string) $otp_data['code'] ) {
            echo json_encode( ['state' => 'error', 'message' => 'Invalid OTP code'] );
            die;
        }

        global $db;

        // Ensure phone columns exist
        $column_check = $db->query( "SHOW COLUMNS FROM " . DB_TABLE_PREFIX . "users LIKE 'phone'" );
        if( $column_check && $column_check->num_rows === 0 ) {
            $db->query( "ALTER TABLE " . DB_TABLE_PREFIX . "users ADD COLUMN phone VARCHAR(20) DEFAULT NULL AFTER email" );
        }
        $verified_check = $db->query( "SHOW COLUMNS FROM " . DB_TABLE_PREFIX . "users LIKE 'phone_verified'" );
        if( $verified_check && $verified_check->num_rows === 0 ) {
            $db->query( "ALTER TABLE " . DB_TABLE_PREFIX . "users ADD COLUMN phone_verified TINYINT(1) NOT NULL DEFAULT 0 AFTER phone" );
        }

        // Prevent verifying a phone that belongs to another user
        $stmt = $db->stmt_init();
        $stmt->prepare( "SELECT id FROM " . DB_TABLE_PREFIX . "users WHERE phone = ? AND id <> ? LIMIT 1" );
        $stmt->bind_param( "si", $phone, $me->ID );
        $stmt->execute();
        $stmt->bind_result( $conflict_id );
        $stmt->fetch();
        $stmt->close();
        if( !empty( $conflict_id ) ) {
            echo json_encode( ['state' => 'error', 'message' => 'This mobile number is already used by another account'] );
            die;
        }

        $stmt = $db->stmt_init();
        $stmt->prepare( "UPDATE " . DB_TABLE_PREFIX . "users SET phone = ?, phone_verified = 1 WHERE id = ?" );
        $stmt->bind_param( "si", $phone, $me->ID );
        $ok = $stmt->execute();
        $stmt->close();

        unset( $_SESSION['otp_profile_verification'][$me->ID] );

        if( $ok ) {
            echo json_encode( ['state' => 'success', 'message' => 'Phone number verified successfully'] );
        } else {
            echo json_encode( ['state' => 'error', 'message' => 'Error updating phone verification status'] );
        }

        die;

    }

    if( $action == 'send_otp' ) {
        
        if( empty( $phone ) || strlen( $phone ) < 10 ) {
            echo json_encode( ['state' => 'error', 'message' => 'Please enter a valid phone number'] );
            die;
        }
        
        // Use Firebase REST API to send OTP
        // Note: This requires Firebase Admin SDK or REST API setup
        // For now, we'll use a simple verification code stored in session
        
        $verification_code = rand( 100000, 999999 );
        $_SESSION['otp_verification'][$phone] = [
            'code' => $verification_code,
            'expires' => time() + 300, // 5 minutes
            'attempts' => 0
        ];
        
        // In production, send SMS via Firebase or SMS gateway
        // For demo, we'll just return the code (remove in production!)
        echo json_encode( [
            'state' => 'success',
            'message' => 'OTP sent to ' . $phone,
            'verification_id' => md5( $phone . $verification_code . time() ),
            'debug_code' => $verification_code // Remove in production!
        ] );
        die;
        
    } elseif( $action == 'verify_otp' ) {
        
        if( empty( $phone ) || empty( $otp ) ) {
            echo json_encode( ['state' => 'error', 'message' => 'Please enter phone number and OTP'] );
            die;
        }
        
        // Verify OTP from session
        if( !isset( $_SESSION['otp_verification'][$phone] ) ) {
            echo json_encode( ['state' => 'error', 'message' => 'OTP session expired. Please request a new OTP'] );
            die;
        }
        
        $otp_data = $_SESSION['otp_verification'][$phone];
        
        if( time() > $otp_data['expires'] ) {
            unset( $_SESSION['otp_verification'][$phone] );
            echo json_encode( ['state' => 'error', 'message' => 'OTP expired. Please request a new OTP'] );
            die;
        }
        
        if( $otp_data['attempts'] >= 5 ) {
            unset( $_SESSION['otp_verification'][$phone] );
            echo json_encode( ['state' => 'error', 'message' => 'Too many attempts. Please request a new OTP'] );
            die;
        }
        
        $_SESSION['otp_verification'][$phone]['attempts']++;
        
        if( $otp != $otp_data['code'] ) {
            echo json_encode( ['state' => 'error', 'message' => 'Invalid OTP code'] );
            die;
        }
        
        // OTP verified - create or login user
        global $db;
        
        // Check if user exists with this phone number
        $stmt = $db->stmt_init();
        $stmt->prepare( "SELECT id, name, email FROM " . DB_TABLE_PREFIX . "users WHERE phone = ? LIMIT 1" );
        $stmt->bind_param( "s", $phone );
        $stmt->execute();
        $stmt->bind_result( $user_id, $user_name, $user_email );
        $user_exists = $stmt->fetch();
        $stmt->close();
        
        if( $user_exists ) {
            // Mark phone as verified (if column exists / auto-add it)
            $verified_check = $db->query( "SHOW COLUMNS FROM " . DB_TABLE_PREFIX . "users LIKE 'phone_verified'" );
            if( $verified_check && $verified_check->num_rows === 0 ) {
                $db->query( "ALTER TABLE " . DB_TABLE_PREFIX . "users ADD COLUMN phone_verified TINYINT(1) NOT NULL DEFAULT 0 AFTER phone" );
            }
            $stmt2 = $db->stmt_init();
            $stmt2->prepare( "UPDATE " . DB_TABLE_PREFIX . "users SET phone_verified = 1 WHERE id = ?" );
            $stmt2->bind_param( "i", $user_id );
            $stmt2->execute();
            $stmt2->close();

            // Login existing user - create session like normal login
            $session = md5( \site\utils::str_random(15) );
            global $db;
            
            // Delete old sessions
            $stmt = $db->stmt_init();
            $stmt->prepare( "DELETE FROM " . DB_TABLE_PREFIX . "sessions WHERE user = ?" );
            $stmt->bind_param( "i", $user_id );
            $stmt->execute();
            
            // Insert new session
            $stmt->prepare( "INSERT INTO " . DB_TABLE_PREFIX . "sessions SET user = ?, session = ?, expiration = DATE_ADD(NOW(), INTERVAL " . DEF_USER_SESSION . " MINUTE), date = NOW()" );
            $stmt->bind_param( "is", $user_id, $session );
            $stmt->execute();
            
            // Update last login
            $stmt->prepare( "UPDATE " . DB_TABLE_PREFIX . "users SET last_login = NOW(), visits = visits + 1 WHERE id = ?" );
            $stmt->bind_param( "i", $user_id );
            $stmt->execute();
            $stmt->close();
            
            $_SESSION['session'] = $session;
            
            // Update name if provided and different
            if( !empty( $name ) && $name != $user_name ) {
                $stmt = $db->stmt_init();
                $stmt->prepare( "UPDATE " . DB_TABLE_PREFIX . "users SET name = ? WHERE id = ?" );
                $stmt->bind_param( "si", $name, $user_id );
                $stmt->execute();
                $stmt->close();
            }
            
            echo json_encode( [
                'state' => 'success',
                'message' => 'Login successful',
                'redirect' => $GLOBALS['siteURL'] . 'setSession.php?back=' . base64_encode( tlink( 'user/account' ) )
            ] );
        } else {
            // Register new user
            if( empty( $name ) || strlen( $name ) < 3 ) {
                echo json_encode( ['state' => 'error', 'message' => 'Please enter your name (minimum 3 characters)'] );
                die;
            }
            
            // Generate email from phone if not exists
            $email = $phone . '@mobile.' . parse_url( $GLOBALS['siteURL'], PHP_URL_HOST );
            
            // Check if email already exists
            $stmt = $db->stmt_init();
            $stmt->prepare( "SELECT id FROM " . DB_TABLE_PREFIX . "users WHERE email = ? LIMIT 1" );
            $stmt->bind_param( "s", $email );
            $stmt->execute();
            if( $stmt->fetch() ) {
                $email = $phone . '_' . time() . '@mobile.' . parse_url( $GLOBALS['siteURL'], PHP_URL_HOST );
            }
            $stmt->close();
            
            // Create user - check if phone column exists, if not add it
            $password = md5( \site\utils::str_random(15) ); // MD5 hash like existing system
            $points = (int) \query\main::get_option( 'u_def_points' );
            $IPaddr = \site\utils::getIP();
            $valid = 1; // Auto-verify mobile users
            $refid = isset( $_COOKIE['referrer'] ) ? (int) $_COOKIE['referrer'] : 0;
            $extra = @serialize( array() );
            
            // Check if phone column exists
            $column_check = $db->query( "SHOW COLUMNS FROM " . DB_TABLE_PREFIX . "users LIKE 'phone'" );
            $phone_column_exists = ( $column_check->num_rows > 0 );
            
            // Ensure phone_verified exists
            $verified_check = $db->query( "SHOW COLUMNS FROM " . DB_TABLE_PREFIX . "users LIKE 'phone_verified'" );
            $phone_verified_exists = ( $verified_check && $verified_check->num_rows > 0 );
            if( !$phone_verified_exists ) {
                $db->query( "ALTER TABLE " . DB_TABLE_PREFIX . "users ADD COLUMN phone_verified TINYINT(1) NOT NULL DEFAULT 0 AFTER phone" );
                $phone_verified_exists = true;
            }

            if( $phone_column_exists ) {
                $stmt = $db->stmt_init();
                if( $phone_verified_exists ) {
                    $stmt->prepare( "INSERT INTO " . DB_TABLE_PREFIX . "users (name, email, password, phone, phone_verified, points, ipaddr, last_action, valid, refid, extra, date) VALUES (?, ?, ?, ?, 1, ?, ?, NOW(), ?, ?, ?, NOW())" );
                    $stmt->bind_param( "sssssiis", $name, $email, $password, $phone, $points, $IPaddr, $valid, $refid, $extra );
                } else {
                    $stmt->prepare( "INSERT INTO " . DB_TABLE_PREFIX . "users (name, email, password, phone, points, ipaddr, last_action, valid, refid, extra, date) VALUES (?, ?, ?, ?, ?, ?, NOW(), ?, ?, ?, NOW())" );
                    $stmt->bind_param( "ssssssiis", $name, $email, $password, $phone, $points, $IPaddr, $valid, $refid, $extra );
                }
            } else {
                // Add phone column if it doesn't exist
                $db->query( "ALTER TABLE " . DB_TABLE_PREFIX . "users ADD COLUMN phone VARCHAR(20) DEFAULT NULL AFTER email" );
                $stmt = $db->stmt_init();
                if( $phone_verified_exists ) {
                    $stmt->prepare( "INSERT INTO " . DB_TABLE_PREFIX . "users (name, email, password, phone, phone_verified, points, ipaddr, last_action, valid, refid, extra, date) VALUES (?, ?, ?, ?, 1, ?, ?, NOW(), ?, ?, ?, NOW())" );
                    $stmt->bind_param( "sssssiis", $name, $email, $password, $phone, $points, $IPaddr, $valid, $refid, $extra );
                } else {
                    $stmt->prepare( "INSERT INTO " . DB_TABLE_PREFIX . "users (name, email, password, phone, points, ipaddr, last_action, valid, refid, extra, date) VALUES (?, ?, ?, ?, ?, ?, NOW(), ?, ?, ?, NOW())" );
                    $stmt->bind_param( "ssssssiis", $name, $email, $password, $phone, $points, $IPaddr, $valid, $refid, $extra );
                }
            }
            
            $stmt->execute();
            $new_user_id = $stmt->insert_id;
            
            if( $new_user_id ) {
                // Create session
                $session = md5( \site\utils::str_random(15) );
                $stmt->prepare( "INSERT INTO " . DB_TABLE_PREFIX . "sessions SET user = ?, session = ?, expiration = DATE_ADD(NOW(), INTERVAL " . DEF_USER_SESSION . " MINUTE), date = NOW()" );
                $stmt->bind_param( "is", $new_user_id, $session );
                $stmt->execute();
                
                $stmt->prepare( "UPDATE " . DB_TABLE_PREFIX . "users SET last_login = NOW(), visits = 1 WHERE id = ?" );
                $stmt->bind_param( "i", $new_user_id );
                $stmt->execute();
                
                $_SESSION['session'] = $session;
                
                echo json_encode( [
                    'state' => 'success',
                    'message' => 'Registration successful',
                    'redirect' => $GLOBALS['siteURL'] . 'setSession.php?back=' . base64_encode( tlink( 'user/account' ) )
                ] );
            } else {
                echo json_encode( ['state' => 'error', 'message' => 'Registration failed. Please try again'] );
            }
        }
        
        // Clear OTP session
        unset( $_SESSION['otp_verification'][$phone] );
        die;
    }
}

// If no action, show OTP form (handled by theme)
die( 'Invalid request' );

