<?php

/**
 * Render Campaign Banners
 * Supports split-screen layout, AdSense, and responsive design
 */
function render_campaign_banners( $position = 'after_slider' ) {
    $campaigns = get_theme_option( 'campaigns' );
    
    if( !is_array( $campaigns ) || empty( $campaigns ) ) {
        return '';
    }
    
    // Filter campaigns by position and expiration
    $filtered_campaigns = array();
    foreach( $campaigns as $campaign ) {
        if( !isset( $campaign['campaign'] ) ) {
            continue;
        }
        
        $camp = $campaign['campaign'];
        
        // Check position
        $camp_position = !empty( $camp['position'] ) ? $camp['position'] : 'after_slider';
        if( $camp_position !== $position ) {
            continue;
        }
        
        // Check expiration
        if( !empty( $camp['remove'] ) && !empty( $camp['expiration'] ) ) {
            if( strtotime( $camp['expiration'] ) < strtotime( 'now' ) ) {
                continue; // Skip expired campaigns
            }
        }
        
        // Must have either image or AdSense code
        if( empty( $camp['image'] ) && empty( $camp['adsense_code'] ) && empty( $camp['image_url'] ) ) {
            continue;
        }
        
        $filtered_campaigns[] = $camp;
    }
    
    if( empty( $filtered_campaigns ) ) {
        return '';
    }
    
    // Limit to 2 campaigns for split-screen layout
    $filtered_campaigns = array_slice( $filtered_campaigns, 0, 2 );
    
    $output = '<div class="campaign-banners-section bgWhite pt-75 pb-75 clearfix">';
    $output .= '<div class="container">';
    $output .= '<div class="campaign-banners-container">';
    
    foreach( $filtered_campaigns as $camp ) {
        $output .= '<div class="campaign-banner-item">';
        
        // Check if AdSense code is provided
        if( !empty( $camp['adsense_code'] ) ) {
            // AdSense banner
            $output .= '<div class="campaign-adsense-wrapper">';
            $output .= $camp['adsense_code'];
            $output .= '</div>';
        } else {
            // Image banner
            $image = '';
            
            // Check if image_url is provided (imported image)
            if( !empty( $camp['image_url'] ) ) {
                $image = $camp['image_url'];
                // If it's a relative path, make it absolute
                if( strpos( $image, 'http' ) !== 0 ) {
                    $image = site_url( $image );
                }
            } else if( !empty( $camp['image'] ) ) {
                $image = $camp['image'];
                // Handle JSON encoded images
                if( !filter_var( $image, FILTER_VALIDATE_URL ) ) {
                    $image_json = @json_decode( $image, true );
                    if( $image_json && is_array( $image_json ) ) {
                        $image = current( $image_json );
                    }
                }
                // If it's a relative path, make it absolute
                if( !filter_var( $image, FILTER_VALIDATE_URL ) && strpos( $image, 'http' ) !== 0 ) {
                    $image = site_url( $image );
                }
            }
            
            if( !empty( $image ) ) {
                $link = !empty( $camp['link'] ) ? esc_url( $camp['link'] ) : '#';
                $title = !empty( $camp['title'] ) ? esc_html( $camp['title'] ) : '';
                
                $output .= '<a href="' . $link . '" class="campaign-banner-link">';
                $output .= '<div class="campaign-banner-image-wrapper">';
                $output .= '<img src="' . esc_url( $image ) . '" alt="' . esc_attr( $title ) . '" class="campaign-banner-image" />';
                
                // Add expiration countdown if set
                if( !empty( $camp['expiration'] ) ) {
                    $output .= '<div class="campaign-exp-date" data-countdown="' . date( 'D M d Y H:i:s O', strtotime( $camp['expiration'] ) ) . '"></div>';
                }
                
                $output .= '</div>';
                
                // Add title if provided
                if( !empty( $title ) ) {
                    $output .= '<div class="campaign-banner-title">' . $title . '</div>';
                }
                
                $output .= '</a>';
            }
        }
        
        $output .= '</div>';
    }
    
    $output .= '</div>'; // campaign-banners-container
    $output .= '</div>'; // container
    $output .= '</div>'; // campaign-banners-section
    
    return $output;
}

