<?php

/**
 * Razorpay Payment Gateway Adapter
 * 
 * This adapter integrates Razorpay payment gateway with the CMS
 * Requires Razorpay API Key ID and Secret Key to be configured
 */

class Payment_Gateway {

/*
INTENT ACTION
*/

public $intent = 'sale';

/*
AMOUNT TO BE PAID
*/

public $total;

/*
PAYMENT DESCRIPTION
*/

public $description;

/*
CREDIT CARD TYPE
*/

public $card_type;

/*
CREDIT CARD FIRST NAME
*/

public $card_fname;

/*
CREDIT CARD LAST NAME
*/

public $card_lname;

/*
CREDIT CARD NUMBER
*/

public $card_number;

/*
CREDIT CARD EXPIRATION MONTH
*/

public $card_month;

/*
CREDIT CARD EXPIRATION YEAR
*/

public $card_year;

/*
CREDIT CARD CVV
*/

public $card_cvv;

/*
ITEMS
*/

public $items;

/*
SUCCESS URL
*/

public $success_url;

/*
CANCEL URL
*/

public $cancel_url;

/*
RAZORPAY API KEY ID
*/

public $api_key_id;

/*
RAZORPAY API SECRET
*/

public $api_secret;

/*
RAZORPAY API MODE (live/sandbox)
*/

public $api_mode;

/*
Construct class
*/

function __construct() {

  $this->api_key_id = \query\main::get_option( 'razorpay_key_id' );
  $this->api_secret = \query\main::get_option( 'razorpay_key_secret' );
  $this->api_mode = strtolower( \query\main::get_option( 'razorpay_mode' ) );
  
  // Default to live mode if not set
  if( empty( $this->api_mode ) ) {
    $this->api_mode = 'live';
  }

  if( empty( $this->api_key_id ) || empty( $this->api_secret ) ) {
    throw new \Exception( 'Razorpay API credentials are not configured. Please set razorpay_key_id and razorpay_key_secret in admin settings (Settings > API / External Accounts).' );
  }

}

public function do_direct() {

  return true;

}

public function direct() {

  // Convert amount to paise (Razorpay uses smallest currency unit)
  // Assuming CURRENCY is INR, if not, adjust accordingly
  $amount_in_paise = (float) $this->total * 100;
  
  // Generate unique order ID
  $order_id = 'order_' . time() . '_' . uniqid();
  
  // Prepare order data
  $order_data = array(
    'amount' => (int) $amount_in_paise,
    'currency' => defined( 'CURRENCY' ) ? CURRENCY : 'INR',
    'receipt' => $order_id,
    'notes' => array(
      'description' => $this->description,
      'items' => json_encode( $this->items )
    )
  );

  // Create order via Razorpay API
  $ch = curl_init();
  curl_setopt( $ch, CURLOPT_URL, 'https://api.razorpay.com/v1/orders' );
  curl_setopt( $ch, CURLOPT_RETURNTRANSFER, 1 );
  curl_setopt( $ch, CURLOPT_POST, 1 );
  curl_setopt( $ch, CURLOPT_POSTFIELDS, json_encode( $order_data ) );
  curl_setopt( $ch, CURLOPT_USERPWD, $this->api_key_id . ':' . $this->api_secret );
  curl_setopt( $ch, CURLOPT_HTTPHEADER, array( 'Content-Type: application/json' ) );

  $response = curl_exec( $ch );
  $http_code = curl_getinfo( $ch, CURLINFO_HTTP_CODE );
  curl_close( $ch );

  if( $http_code !== 200 ) {
    $error_data = json_decode( $response, true );
    $error_msg = isset( $error_data['error']['description'] ) ? $error_data['error']['description'] : 'Failed to create Razorpay order';
    throw new \Exception( 'Razorpay error: ' . $error_msg );
  }

  $order_response = json_decode( $response, true );

  if( !isset( $order_response['id'] ) ) {
    throw new \Exception( 'Razorpay error: Invalid response from API' );
  }

  // Build Razorpay Checkout URL
  $checkout_url = 'https://checkout.razorpay.com/v1/checkout.js';
  
  // Return payment details for redirect
  // The payment page will need to handle the Razorpay checkout integration
  return array(
    'id' => $order_response['id'],
    'total' => $this->total,
    'items' => $this->items,
    'details' => $this->description,
    'state' => 'created',
    'href' => $this->success_url . '&razorpay_order_id=' . $order_response['id'] . '&razorpay_amount=' . $amount_in_paise,
    'razorpay_order_id' => $order_response['id'],
    'razorpay_key_id' => $this->api_key_id,
    'razorpay_amount' => $amount_in_paise,
    'razorpay_currency' => $order_response['currency'],
    'razorpay_name' => defined( 'SITE_NAME' ) ? SITE_NAME : 'Payment',
    'razorpay_description' => $this->description,
    'razorpay_prefill_email' => isset( $GLOBALS['me']->Email ) ? $GLOBALS['me']->Email : '',
    'razorpay_prefill_contact' => isset( $GLOBALS['me']->Phone ) ? $GLOBALS['me']->Phone : '',
    'razorpay_cancel_url' => $this->cancel_url
  );

}

public function do_credit_card() {

  // Razorpay supports credit card payments through their checkout
  // but we'll use direct payment method for better UX
  return false;

}

public function credit_card() {

  // Not used for Razorpay as it handles cards through checkout
  throw new \Exception( 'Credit card payment not available for Razorpay. Please use direct payment.' );

}

public function execute_direct_payment() {

  // Check if payment was successful via Razorpay callback
  if( isset( $_GET['razorpay_payment_id'] ) && isset( $_GET['razorpay_order_id'] ) && isset( $_GET['razorpay_signature'] ) ) {
    
    // Verify payment signature
    $payment_id = $_GET['razorpay_payment_id'];
    $order_id = $_GET['razorpay_order_id'];
    $signature = $_GET['razorpay_signature'];
    
    // Generate expected signature
    $payload = $order_id . '|' . $payment_id;
    $expected_signature = hash_hmac( 'sha256', $payload, $this->api_secret );
    
    if( $expected_signature === $signature ) {
      return $payment_id;
    }
    
  }
  
  // Also check for payment_id in POST (webhook callback)
  if( isset( $_POST['razorpay_payment_id'] ) && isset( $_POST['razorpay_order_id'] ) && isset( $_POST['razorpay_signature'] ) ) {
    
    $payment_id = $_POST['razorpay_payment_id'];
    $order_id = $_POST['razorpay_order_id'];
    $signature = $_POST['razorpay_signature'];
    
    $payload = $order_id . '|' . $payment_id;
    $expected_signature = hash_hmac( 'sha256', $payload, $this->api_secret );
    
    if( $expected_signature === $signature ) {
      return $payment_id;
    }
    
  }

  return false;

}

public function execute_payment() {

  $payment_id = $this->execute_direct_payment();
  
  if( !$payment_id ) {
    throw new \Exception( 'Razorpay payment verification failed.' );
  }

  // Fetch payment details from Razorpay API
  $ch = curl_init();
  curl_setopt( $ch, CURLOPT_URL, 'https://api.razorpay.com/v1/payments/' . $payment_id );
  curl_setopt( $ch, CURLOPT_RETURNTRANSFER, 1 );
  curl_setopt( $ch, CURLOPT_USERPWD, $this->api_key_id . ':' . $this->api_secret );
  curl_setopt( $ch, CURLOPT_HTTPHEADER, array( 'Content-Type: application/json' ) );

  $response = curl_exec( $ch );
  $http_code = curl_getinfo( $ch, CURLINFO_HTTP_CODE );
  curl_close( $ch );

  if( $http_code !== 200 ) {
    throw new \Exception( 'Razorpay error: Failed to fetch payment details' );
  }

  $payment_data = json_decode( $response, true );

  if( !isset( $payment_data['status'] ) ) {
    throw new \Exception( 'Razorpay error: Invalid payment response' );
  }

  // Map Razorpay status to our state
  $state = 'pending';
  if( $payment_data['status'] === 'authorized' || $payment_data['status'] === 'captured' ) {
    $state = 'completed';
  } else if( $payment_data['status'] === 'failed' ) {
    $state = 'failed';
  }

  return array(
    'id' => $payment_id,
    'state' => $state,
    'razorpay_status' => $payment_data['status']
  );

}

}

