# Complete CDN Setup Guide

## 🎯 Overview
ALL CSS and JavaScript files are now centralized in 4 include files for easy CDN management.

---

## 📁 Centralized Files:

### 1. **views/includes/header-css.php**
All CSS files loaded in `<head>` section (37 CSS files total)

### 2. **views/includes/header-js.php**  
All JavaScript files loaded in `<head>` section (11 JS files)

### 3. **views/includes/footer-css.php**
CSS files loaded before `</body>` (2 CSS files)

### 4. **views/includes/footer-js.php**
JavaScript files loaded before `</body>` (1 JS file)

---

## 🚀 Quick Start - Enable CDN in 4 Steps:

### Step 1: Upload Files to Your CDN

Upload all CSS files to your CDN:
```
assets/css/*.css  → Upload to: https://your-cdn.com/css/
assets/js/*.js    → Upload to: https://your-cdn.com/js/
```

### Step 2: Set CDN URLs

Edit these 4 files and change line 8 in each:

#### **views/includes/header-css.php** (Line 8)
```php
$cdn_css_url = 'https://your-cdn.com/css';
```

#### **views/includes/header-js.php** (Line 8)
```php
$cdn_js_url = 'https://your-cdn.com/js';
```

#### **views/includes/footer-css.php** (Line 8)
```php
$cdn_css_url = 'https://your-cdn.com/css';
```

#### **views/includes/footer-js.php** (Line 8)
```php
$cdn_js_url = 'https://your-cdn.com/js';
```

### Step 3: Test
1. Clear browser cache
2. Visit your website
3. Open DevTools → Network tab
4. Verify files load from CDN

### Step 4: Done!
All pages now load CSS/JS from your CDN!

---

## 📋 Complete File List:

### Header CSS Files (37 files):
```
tabler-icons.min.css
styles.css
swiper-custom.css
swiper-custom-social.css
compact-header.css
category-menu.css
ajax-search.css
quintessential-font.css
animated-placeholder.css
auth-pages.css
campaign-styles.css
campaigns-slider.css
categories-slider.css
featured-categories.css
sidebar-campaigns.css
404-page.css
store-cards.css
footer-categories.css
page-content.css
footer-logo.css
popup-system.css
push-notifications.css
coupon-filter.css
css-stars.css
site-fixes.css
notification-bell.css
filter-sidebar-redesign.css
sidenav-redesign.css
categories-slider-redesign.css
homepage-featured-offers.css
campaigns-redesign.css
why-choose-us.css
coupon-modal-redesign.css
```

### Header JS Files (11 files):
```
jquery.js
uikit.js
uikit-icons.js
swiper-init.js
sticky-header.js
scroll-to-top.js
ajax-search.js
animated-placeholder.js
popup-system.js
push-notifications.js
coupon-filter.js
```

### Footer CSS Files (2 files):
```
footer-redesign.css
modern-footer-newsletter.css
```

### Footer JS Files (1 file):
```
modern-footer-newsletter.js
notification-bell.js
```

---

## 📍 Where Files Are Included:

### Header
**File:** `views/header.view.php`  
**Lines:** ~70-75
```php
<!-- Centralized CSS -->
<?php include __DIR__ . '/includes/header-css.php'; ?>

<!-- Centralized JS -->
<?php include __DIR__ . '/includes/header-js.php'; ?>
```

### Footer
**File:** `sections/views/footer.view.php`  
**Lines:** ~150-152
```php
<!-- Centralized Footer CSS & JS -->
<?php include __DIR__ . '/../includes/footer-css.php'; ?>
<?php include __DIR__ . '/../includes/footer-js.php'; ?>
```

---

## 🔄 Switching Between Local and CDN:

### Use CDN:
```php
// In all 4 include files, set CDN URL:
$cdn_css_url = 'https://cdn.example.com/css';
$cdn_js_url = 'https://cdn.example.com/js';
```

### Use Local Files:
```php
// In all 4 include files, leave empty:
$cdn_css_url = '';
$cdn_js_url = '';
```

---

## 📝 Example CDN URLs:

### Cloudflare R2:
```php
$cdn_css_url = 'https://pub-abc123.r2.dev/css';
$cdn_js_url = 'https://pub-abc123.r2.dev/js';
```

### Amazon S3:
```php
$cdn_css_url = 'https://mybucket.s3.amazonaws.com/css';
$cdn_js_url = 'https://mybucket.s3.amazonaws.com/js';
```

### Custom CDN:
```php
$cdn_css_url = 'https://cdn.yourdomain.com/assets/css';
$cdn_js_url = 'https://cdn.yourdomain.com/assets/js';
```

### BunnyCDN:
```php
$cdn_css_url = 'https://yourzone.b-cdn.net/css';
$cdn_js_url = 'https://yourzone.b-cdn.net/js';
```

---

## ✅ Benefits:

1. **Single Point of Control** - Change CDN URL in 4 places, affects entire site
2. **Easy Switching** - Toggle between local and CDN instantly
3. **Clean Code** - No duplicate CSS/JS links scattered across files
4. **Fast Updates** - Update all pages at once
5. **Version Control** - Cache busting preserved with `?v=<?php echo time(); ?>`
6. **Performance** - Load assets from CDN for faster page loads
7. **Bandwidth Savings** - Reduce server bandwidth usage

---

## 🛠️ Advanced Configuration:

### Different CDN for CSS and JS:
```php
// header-css.php
$cdn_css_url = 'https://css-cdn.example.com';

// header-js.php
$cdn_js_url = 'https://js-cdn.example.com';
```

### Add Version/Cache Busting:
```php
// In include files, modify helper function:
function getCssUrl($filename) {
    global $cdn_css_url, $urlPath;
    if (!empty($cdn_css_url)) {
        return $cdn_css_url . '/' . $filename . '?v=' . time();
    }
    return $urlPath->assets_css($filename);
}
```

### Add New CSS/JS File:
1. Upload file to CDN
2. Edit appropriate include file
3. Add line using helper function:
```php
<link rel="stylesheet" href="<?php echo getCssUrl('your-new-file.css'); ?>">
```

---

## 🔍 Troubleshooting:

### CSS/JS Not Loading?
1. ✅ Check CDN URL is correct (no trailing slash)
2. ✅ Verify files uploaded to CDN
3. ✅ Check file permissions (public read)
4. ✅ Clear browser cache (Ctrl+Shift+R)
5. ✅ Check browser console for 404 errors
6. ✅ Verify CDN CORS settings

### Mixed Content Warnings?
- Ensure CDN URL uses HTTPS, not HTTP

### Files Loading from Wrong Location?
- Check if CDN URL variable is set correctly
- Verify no typos in CDN URL
- Clear server-side cache if using caching

### Want to Test CDN Before Going Live?
```php
// Use conditional CDN based on environment
$cdn_css_url = ($_SERVER['HTTP_HOST'] == 'localhost') ? '' : 'https://cdn.example.com/css';
```

---

## 📊 File Structure:

```
views/
├── includes/
│   ├── header-css.php    ← 37 CSS files
│   ├── header-js.php     ← 11 JS files
│   ├── footer-css.php    ← 2 CSS files
│   └── footer-js.php     ← 2 JS files
├── header.view.php       ← Includes header-css.php & header-js.php
└── ...

sections/
└── views/
    └── footer.view.php   ← Includes footer-css.php & footer-js.php
```

---

## 🎯 Summary:

| What | Where | Files Count |
|------|-------|-------------|
| Header CSS | `views/includes/header-css.php` | 37 CSS files |
| Header JS | `views/includes/header-js.php` | 11 JS files |
| Footer CSS | `views/includes/footer-css.php` | 2 CSS files |
| Footer JS | `views/includes/footer-js.php` | 2 JS files |
| **TOTAL** | **4 include files** | **52 files** |

---

## 🚀 Performance Tips:

1. **Enable Gzip/Brotli** on your CDN
2. **Set Long Cache Headers** (1 year for static assets)
3. **Use HTTP/2** for multiplexing
4. **Minify CSS/JS** before uploading to CDN
5. **Combine Files** if possible (reduce HTTP requests)
6. **Use CDN with Edge Locations** close to your users

---

## 📌 Important Notes:

- ✅ All individual CSS/JS links removed from section files
- ✅ Everything centralized in 4 include files
- ✅ Helper functions automatically switch between CDN and local
- ✅ Cache busting preserved where needed
- ✅ External CDN links (Font Awesome, Swiper, Google Fonts) unchanged
- ✅ Backward compatible - empty CDN URL = use local files
- ✅ RTL CSS files remain separate (loaded conditionally)
- ✅ Google Analytics, Firebase, reCAPTCHA scripts unchanged

---

**That's it!** Complete centralized CDN management for all your assets in just 4 files!

Need help? Just edit the 4 include files and set your CDN URL. Everything else is automatic! 🎉
