<?php
/**
 * SEO Features Dashboard
 * Manage Image SEO, Internal Linking, and Performance
 */

session_start();
require '../../config.php';
require '../admin_config.php';

// Check admin access
if (!isset($_SESSION['user_email']) || $_SESSION['user_role'] != 1) {
    header('Location: ../login.php');
    exit;
}

require '../../includes/image_seo_optimizer.php';
require '../../includes/internal_linking_engine.php';
require '../../includes/performance_optimizer.php';

// Initialize classes
$imageSEO = new ImageSEOOptimizer($connect);
$linkEngine = new InternalLinkingEngine($connect);
$perfOptimizer = new PerformanceOptimizer();

// Get statistics
$imageAudit = $imageSEO->auditImages();
$linkingAnalysis = $linkEngine->analyzeLinkingStructure();

// Page title
$pageTitle = 'SEO Features Dashboard';

// Include header
require '../views/header.view.php';
require '../views/sidebar.view.php';
?>

<div class="uk-container uk-margin-top">
    <h1 class="uk-heading-line"><span>🚀 SEO Features Dashboard</span></h1>
    
    <!-- Stats Cards -->
    <div class="uk-grid-small uk-child-width-1-3@m" uk-grid>
        <!-- Image SEO Stats -->
        <div>
            <div class="uk-card uk-card-default uk-card-body">
                <h3 class="uk-card-title">📸 Image SEO</h3>
                <p class="uk-text-large uk-text-bold"><?php echo count($imageAudit); ?></p>
                <p class="uk-text-muted">Images need optimization</p>
                <a href="#image-audit" uk-scroll class="uk-button uk-button-primary uk-button-small">View Details</a>
            </div>
        </div>
        
        <!-- Internal Linking Stats -->
        <div>
            <div class="uk-card uk-card-default uk-card-body">
                <h3 class="uk-card-title">🔗 Internal Linking</h3>
                <p class="uk-text-large uk-text-bold"><?php echo $linkingAnalysis['avg_links_per_page']; ?></p>
                <p class="uk-text-muted">Avg links per page</p>
                <a href="#linking-analysis" uk-scroll class="uk-button uk-button-primary uk-button-small">View Details</a>
            </div>
        </div>
        
        <!-- Performance Stats -->
        <div>
            <div class="uk-card uk-card-default uk-card-body">
                <h3 class="uk-card-title">⚡ Performance</h3>
                <p class="uk-text-large uk-text-bold">Active</p>
                <p class="uk-text-muted">Optimization enabled</p>
                <a href="https://pagespeed.web.dev/?url=https://alonedeals.com" target="_blank" class="uk-button uk-button-primary uk-button-small">Test Speed</a>
            </div>
        </div>
    </div>

    
    <!-- Image Audit Section -->
    <div id="image-audit" class="uk-margin-large-top">
        <h2>📸 Image SEO Audit</h2>
        <div class="uk-card uk-card-default uk-card-body">
            <?php if (empty($imageAudit)): ?>
                <p class="uk-text-success">✅ All images are optimized!</p>
            <?php else: ?>
                <p class="uk-text-warning">⚠️ Found <?php echo count($imageAudit); ?> images that need optimization</p>
                <table class="uk-table uk-table-striped uk-table-small">
                    <thead>
                        <tr>
                            <th>Type</th>
                            <th>Title</th>
                            <th>Issue</th>
                            <th>Action</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach (array_slice($imageAudit, 0, 10) as $issue): ?>
                        <tr>
                            <td><?php echo ucfirst($issue['type']); ?></td>
                            <td><?php echo htmlspecialchars($issue['title']); ?></td>
                            <td><?php echo htmlspecialchars($issue['issue']); ?></td>
                            <td>
                                <?php if ($issue['type'] == 'blog_post'): ?>
                                    <a href="../controller/edit_blog.php?id=<?php echo $issue['id']; ?>" class="uk-button uk-button-small uk-button-primary">Edit</a>
                                <?php endif; ?>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
                <?php if (count($imageAudit) > 10): ?>
                    <p class="uk-text-muted">Showing 10 of <?php echo count($imageAudit); ?> issues</p>
                <?php endif; ?>
            <?php endif; ?>
        </div>
    </div>
    
    <!-- Internal Linking Analysis -->
    <div id="linking-analysis" class="uk-margin-large-top">
        <h2>🔗 Internal Linking Analysis</h2>
        <div class="uk-card uk-card-default uk-card-body">
            <div class="uk-grid-small" uk-grid>
                <div class="uk-width-1-3@m">
                    <div class="uk-card uk-card-primary uk-card-body uk-text-center">
                        <p class="uk-text-large uk-margin-remove"><?php echo $linkingAnalysis['total_pages']; ?></p>
                        <p class="uk-text-small uk-margin-remove">Total Pages</p>
                    </div>
                </div>
                <div class="uk-width-1-3@m">
                    <div class="uk-card uk-card-secondary uk-card-body uk-text-center">
                        <p class="uk-text-large uk-margin-remove"><?php echo $linkingAnalysis['pages_with_links']; ?></p>
                        <p class="uk-text-small uk-margin-remove">Pages with Links</p>
                    </div>
                </div>
                <div class="uk-width-1-3@m">
                    <div class="uk-card uk-card-default uk-card-body uk-text-center">
                        <p class="uk-text-large uk-margin-remove"><?php echo $linkingAnalysis['avg_links_per_page']; ?></p>
                        <p class="uk-text-small uk-margin-remove">Avg Links/Page</p>
                    </div>
                </div>
            </div>
            
            <?php if (!empty($linkingAnalysis['orphan_pages'])): ?>
                <hr>
                <h3>Orphan Pages (No Internal Links)</h3>
                <ul class="uk-list uk-list-striped">
                    <?php foreach (array_slice($linkingAnalysis['orphan_pages'], 0, 5) as $page): ?>
                        <li>
                            <?php echo htmlspecialchars($page['title']); ?>
                            <span class="uk-badge"><?php echo $page['type']; ?></span>
                        </li>
                    <?php endforeach; ?>
                </ul>
            <?php endif; ?>
        </div>
    </div>
    
    <!-- Performance Optimization -->
    <div class="uk-margin-large-top">
        <h2>⚡ Performance Optimization</h2>
        <div class="uk-card uk-card-default uk-card-body">
            <div class="uk-alert uk-alert-success">
                <p>✅ Performance optimization is active on your site!</p>
            </div>
            
            <h3>Active Features:</h3>
            <ul class="uk-list uk-list-bullet">
                <li>✅ Preconnect to external domains</li>
                <li>✅ DNS prefetch for faster loading</li>
                <li>✅ Image lazy loading</li>
                <li>✅ Async image decoding</li>
                <li>✅ Related content widgets</li>
                <li>✅ Auto internal linking</li>
            </ul>
            
            <h3>Next Steps:</h3>
            <ol class="uk-list uk-list-decimal">
                <li>Add .htaccess rules from <code>.htaccess_performance_rules.txt</code></li>
                <li>Test your site speed at <a href="https://pagespeed.web.dev/?url=https://alonedeals.com" target="_blank">PageSpeed Insights</a></li>
                <li>Monitor Core Web Vitals in Google Search Console</li>
            </ol>
            
            <a href="https://pagespeed.web.dev/?url=https://alonedeals.com" target="_blank" class="uk-button uk-button-primary">
                Test Site Speed
            </a>
        </div>
    </div>
    
    <!-- Documentation -->
    <div class="uk-margin-large-top uk-margin-large-bottom">
        <h2>📚 Documentation</h2>
        <div class="uk-card uk-card-default uk-card-body">
            <p>For complete documentation and usage instructions, see:</p>
            <ul class="uk-list uk-list-bullet">
                <li><strong>SEO_FEATURES_INTEGRATED.md</strong> - Complete integration guide</li>
                <li><strong>FINAL_SEO_FEATURES.md</strong> - Technical documentation</li>
                <li><strong>.htaccess_performance_rules.txt</strong> - Performance rules</li>
            </ul>
        </div>
    </div>
</div>

<?php require '../views/footer.view.php'; ?>
