<?php
// CACHE DIAGNOSTIC TOOL
// This page shows the ACTUAL database state vs what might be cached

// Prevent caching
header("Cache-Control: no-store, no-cache, must-revalidate, max-age=0");
header("Pragma: no-cache");
header("Expires: 0");

require '../config.php';
require 'functions.php';

$connect = connect($database);

// Get store ID from URL
$store_name = isset($_GET['store']) ? $_GET['store'] : 'Shopsy';

// Find store by name
$stmt = $connect->prepare("SELECT * FROM stores WHERE store_title LIKE :name LIMIT 1");
$stmt->execute([':name' => '%' . $store_name . '%']);
$store = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$store) {
    die("Store not found: " . htmlspecialchars($store_name));
}

// Get store categories from junction table
$cat_stmt = $connect->prepare("SELECT sc.category_id, c.category_title, c.category_parent 
    FROM store_categories sc 
    LEFT JOIN categories c ON sc.category_id = c.category_id 
    WHERE sc.store_id = :store_id");
$cat_stmt->execute([':store_id' => $store['store_id']]);
$categories = $cat_stmt->fetchAll(PDO::FETCH_ASSOC);

// Get all categories for reference
$all_cats = $connect->query("SELECT * FROM categories ORDER BY category_title")->fetchAll(PDO::FETCH_ASSOC);

?>
<!DOCTYPE html>
<html>
<head>
    <title>Store Cache Diagnostic</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; background: #f5f5f5; }
        .container { max-width: 800px; margin: 0 auto; background: white; padding: 20px; border-radius: 8px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
        h1 { color: #ed174f; border-bottom: 3px solid #ed174f; padding-bottom: 10px; }
        h2 { color: #333; margin-top: 30px; }
        .info-box { background: #f9f9f9; border-left: 4px solid #ed174f; padding: 15px; margin: 15px 0; }
        .success { background: #d4edda; border-color: #28a745; color: #155724; }
        .warning { background: #fff3cd; border-color: #ffc107; color: #856404; }
        table { width: 100%; border-collapse: collapse; margin: 15px 0; }
        th, td { padding: 12px; text-align: left; border-bottom: 1px solid #ddd; }
        th { background: #ed174f; color: white; }
        tr:hover { background: #f5f5f5; }
        .timestamp { color: #666; font-size: 12px; margin-top: 20px; text-align: center; }
        .btn { display: inline-block; padding: 10px 20px; background: #ed174f; color: white; text-decoration: none; border-radius: 5px; margin: 5px; }
        .btn:hover { background: #c01340; }
    </style>
</head>
<body>
    <div class="container">
        <h1>🔍 Store Cache Diagnostic</h1>
        
        <div class="info-box">
            <strong>Store:</strong> <?php echo htmlspecialchars($store['store_title']); ?><br>
            <strong>Store ID:</strong> <?php echo $store['store_id']; ?><br>
            <strong>Legacy Category Field:</strong> <?php echo $store['store_category']; ?>
        </div>

        <h2>📊 Current Database State (store_categories table)</h2>
        <div class="info-box success">
            <strong>✓ This is the ACTUAL data from the database right now</strong><br>
            Generated at: <?php echo date('Y-m-d H:i:s'); ?>
        </div>

        <?php if (empty($categories)): ?>
            <div class="info-box warning">
                ⚠️ No categories assigned to this store in the junction table!
            </div>
        <?php else: ?>
            <table>
                <thead>
                    <tr>
                        <th>Category ID</th>
                        <th>Category Name</th>
                        <th>Type</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($categories as $cat): ?>
                        <tr>
                            <td><?php echo $cat['category_id']; ?></td>
                            <td><?php echo htmlspecialchars($cat['category_title']); ?></td>
                            <td><?php echo $cat['category_parent'] == 0 ? 'Parent Category' : 'Subcategory'; ?></td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        <?php endif; ?>

        <h2>📋 All Available Categories</h2>
        <table>
            <thead>
                <tr>
                    <th>ID</th>
                    <th>Category Name</th>
                    <th>Type</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($all_cats as $cat): ?>
                    <tr>
                        <td><?php echo $cat['category_id']; ?></td>
                        <td><?php echo htmlspecialchars($cat['category_title']); ?></td>
                        <td><?php echo $cat['category_parent'] == 0 ? 'Parent' : 'Subcategory (Parent: ' . $cat['category_parent'] . ')'; ?></td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>

        <h2>🔧 Actions</h2>
        <a href="controller/edit_store.php?id=<?php echo $store['store_id']; ?>" class="btn">Edit This Store</a>
        <a href="check_store_cache.php?store=<?php echo urlencode($store_name); ?>&t=<?php echo time(); ?>" class="btn">Refresh This Page</a>
        
        <div class="timestamp">
            Page generated at: <?php echo date('Y-m-d H:i:s'); ?> | Timestamp: <?php echo time(); ?>
        </div>

        <div class="info-box" style="margin-top: 30px;">
            <strong>💡 Troubleshooting Tips:</strong><br>
            1. If you see old data here, the database hasn't been updated<br>
            2. If you see correct data here but wrong data in admin panel, it's a browser cache issue<br>
            3. Try: Ctrl+Shift+Delete to clear browser cache, or use Incognito mode<br>
            4. Try: Ctrl+F5 for hard refresh on the edit store page<br>
            5. Check browser console (F12) for the debug logs showing selected categories
        </div>
    </div>
</body>
</html>
