<?php
session_start();
if (!isset($_SESSION['user_email'])){
    header('Location: ./login.php');
    exit;
}

require '../config.php';
require './functions.php';

$connect = connect($database);

// Get store ID from URL
$store_id = isset($_GET['id']) ? (int)$_GET['id'] : 887;

// Fetch store data
$query = $connect->prepare("SELECT * FROM stores WHERE store_id = ?");
$query->execute([$store_id]);
$store = $query->fetch(PDO::FETCH_ASSOC);

if (!$store) {
    die("Store not found!");
}

?>
<!DOCTYPE html>
<html>
<head>
    <title>Store Image Debug - ID: <?php echo $store_id; ?></title>
    <style>
        body { font-family: Arial, sans-serif; padding: 20px; background: #f5f5f5; }
        .container { max-width: 800px; margin: 0 auto; background: white; padding: 20px; border-radius: 8px; }
        h1 { color: #333; border-bottom: 2px solid #007bff; padding-bottom: 10px; }
        .section { margin: 20px 0; padding: 15px; background: #f8f9fa; border-left: 4px solid #007bff; }
        .label { font-weight: bold; color: #555; }
        .value { color: #000; margin-left: 10px; }
        .success { color: green; }
        .error { color: red; }
        .warning { color: orange; }
        img { max-width: 200px; border: 2px solid #ddd; margin: 10px 0; }
        code { background: #e9ecef; padding: 2px 6px; border-radius: 3px; }
    </style>
</head>
<body>
    <div class="container">
        <h1>🔍 Store Image Debug Tool</h1>
        
        <div class="section">
            <h2>Store Information</h2>
            <p><span class="label">Store ID:</span><span class="value"><?php echo $store['store_id']; ?></span></p>
            <p><span class="label">Store Name:</span><span class="value"><?php echo htmlspecialchars($store['store_title']); ?></span></p>
        </div>

        <div class="section">
            <h2>Database Values</h2>
            <p><span class="label">Local Image:</span><span class="value"><code><?php echo htmlspecialchars($store['store_image']); ?></code></span></p>
            <p><span class="label">CDN URL:</span><span class="value"><code><?php echo htmlspecialchars($store['store_image_cdn_url'] ?? 'NULL'); ?></code></span></p>
            <p><span class="label">Use CDN:</span><span class="value"><code><?php echo $store['store_use_cdn'] ?? '0'; ?></code> 
                <?php if (isset($store['store_use_cdn']) && $store['store_use_cdn'] == 1): ?>
                    <span class="success">✓ Enabled</span>
                <?php else: ?>
                    <span class="error">✗ Disabled</span>
                <?php endif; ?>
            </span></p>
        </div>

        <div class="section">
            <h2>getImageUrl() Function Test</h2>
            <?php
            // Load the getImageUrl function
            require '../functions.php';
            
            $cdn_url = $store['store_image_cdn_url'] ?? '';
            $use_cdn = $store['store_use_cdn'] ?? 0;
            $local_image = $store['store_image'];
            
            $result_url = getImageUrl($cdn_url, $use_cdn, $local_image);
            ?>
            <p><span class="label">Function Result:</span><span class="value"><code><?php echo htmlspecialchars($result_url); ?></code></span></p>
            
            <h3>Logic Check:</h3>
            <?php if ($use_cdn == 1): ?>
                <p class="success">✓ CDN is enabled (use_cdn = 1)</p>
                <?php if (!empty($cdn_url)): ?>
                    <p class="success">✓ CDN URL is set</p>
                    <?php if (filter_var($cdn_url, FILTER_VALIDATE_URL)): ?>
                        <p class="success">✓ CDN URL is valid</p>
                        <p><strong>Result:</strong> Should use CDN URL</p>
                    <?php else: ?>
                        <p class="error">✗ CDN URL is NOT valid</p>
                        <p><strong>Result:</strong> Will use local image</p>
                    <?php endif; ?>
                <?php else: ?>
                    <p class="error">✗ CDN URL is empty</p>
                    <p><strong>Result:</strong> Will use local image</p>
                <?php endif; ?>
            <?php else: ?>
                <p class="warning">⚠ CDN is disabled (use_cdn = 0)</p>
                <p><strong>Result:</strong> Will use local image</p>
            <?php endif; ?>
        </div>

        <div class="section">
            <h2>Image Preview</h2>
            <h3>Current Result:</h3>
            <img src="<?php echo $result_url; ?>" alt="Store Image" onerror="this.style.border='2px solid red'; this.alt='❌ Image failed to load';">
            <p><small>URL: <code><?php echo htmlspecialchars($result_url); ?></code></small></p>
            
            <?php if (!empty($cdn_url)): ?>
            <h3>CDN URL Direct Test:</h3>
            <img src="<?php echo htmlspecialchars($cdn_url); ?>" alt="CDN Image" onerror="this.style.border='2px solid red'; this.alt='❌ CDN Image failed to load';">
            <p><small>URL: <code><?php echo htmlspecialchars($cdn_url); ?></code></small></p>
            <?php endif; ?>
            
            <h3>Local Image:</h3>
            <img src="<?php echo SITE_URL . '/images/' . $local_image; ?>" alt="Local Image" onerror="this.style.border='2px solid red'; this.alt='❌ Local Image failed to load';">
            <p><small>URL: <code><?php echo htmlspecialchars(SITE_URL . '/images/' . $local_image); ?></code></small></p>
        </div>

        <div class="section">
            <h2>Fix Instructions</h2>
            <?php if ($use_cdn == 0 && !empty($cdn_url)): ?>
                <p class="warning"><strong>⚠ Issue Found:</strong> You have a CDN URL but "Use CDN" is disabled!</p>
                <p><strong>Solution:</strong></p>
                <ol>
                    <li>Go to <a href="./edit_store.php?id=<?php echo $store_id; ?>">Edit Store</a></li>
                    <li>Scroll to "CDN Image Settings"</li>
                    <li>Check the "Use CDN Image" checkbox</li>
                    <li>Click "Update"</li>
                </ol>
            <?php elseif (empty($cdn_url)): ?>
                <p class="error"><strong>✗ Issue Found:</strong> No CDN URL is set!</p>
                <p><strong>Solution:</strong></p>
                <ol>
                    <li>Go to <a href="./edit_store.php?id=<?php echo $store_id; ?>">Edit Store</a></li>
                    <li>Scroll to "CDN Image Settings"</li>
                    <li>Enter your CDN URL in the "CDN Image URL" field</li>
                    <li>Check the "Use CDN Image" checkbox</li>
                    <li>Click "Update"</li>
                </ol>
            <?php elseif (!filter_var($cdn_url, FILTER_VALIDATE_URL)): ?>
                <p class="error"><strong>✗ Issue Found:</strong> CDN URL is not valid!</p>
                <p><strong>Current URL:</strong> <code><?php echo htmlspecialchars($cdn_url); ?></code></p>
                <p><strong>Solution:</strong> Make sure the URL starts with http:// or https://</p>
            <?php else: ?>
                <p class="success"><strong>✓ Configuration looks correct!</strong></p>
                <p>If the image still doesn't show, check:</p>
                <ul>
                    <li>The CDN URL is accessible (not blocked by CORS or firewall)</li>
                    <li>The image file exists at the CDN URL</li>
                    <li>Clear your browser cache (Ctrl+Shift+Delete)</li>
                </ul>
            <?php endif; ?>
        </div>

        <div class="section">
            <p><a href="./stores.php">← Back to Stores</a> | <a href="./edit_store.php?id=<?php echo $store_id; ?>">Edit This Store</a></p>
        </div>
    </div>
</body>
</html>
