# Emergency Fix - Site Not Loading / Footer Missing

## The Problem
1. **Preloader keeps spinning** - JavaScript not loading to remove it
2. **Footer not showing** - Page stops rendering due to missing JS/CSS files
3. **Files deleted from CDN** - Site trying to load from CDN but files don't exist

## Quick Fix Steps

### Step 1: Add Error Display to index.php

Edit `index.php` and add this at the very top:

```php
<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);

require "core.php";
// ... rest of code
?>
```

### Step 2: Check Browser Console

1. Press **F12** to open Developer Tools
2. Go to **Console** tab
3. Look for errors (red text)
4. Common errors:
   - `Failed to load resource` - File not found (404)
   - `$ is not defined` - jQuery not loaded
   - `CORS error` - CDN access blocked

### Step 3: Check Network Tab

1. Press **F12**
2. Go to **Network** tab
3. Reload page
4. Look for **red items** (failed requests)
5. Check if files are loading from:
   - ❌ `https://assets.alonedeals.com` (CDN - files deleted)
   - ✅ `https://www.alonedeals.com` (Local - should work)

### Step 4: Temporarily Disable Preloader

Add this to `views/header.view.php` right after `<body>` tag:

```html
<style>
#preloader {
    display: none !important;
}
</style>
```

This will hide the preloader immediately so you can see the page.

### Step 5: Check All CDN Settings

Make sure these files have CDN disabled:

**views/includes/header-css.php:**
```php
$cdn_css_url = ''; // Must be EMPTY
```

**views/includes/header-js.php:**
```php
$cdn_js_url = ''; // Must be EMPTY
```

**views/includes/footer-js.php:**
```php
$cdn_js_url = ''; // Must be EMPTY
```

**views/includes/footer-css.php (if exists):**
```php
$cdn_css_url = ''; // Must be EMPTY
```

### Step 6: Clear ALL Caches

1. **Browser Cache:**
   - Press `Ctrl+Shift+Delete`
   - Select "All time"
   - Check "Cached images and files"
   - Click "Clear data"

2. **Server Cache:**
   - Access: `https://www.alonedeals.com/clear_server_cache.php`

3. **Hard Refresh:**
   - Press `Ctrl+F5` (Windows)
   - Press `Cmd+Shift+R` (Mac)

### Step 7: Test in Incognito Mode

1. Press `Ctrl+Shift+N` (Chrome) or `Ctrl+Shift+P` (Firefox)
2. Visit your site
3. If it works in Incognito, it's a cache issue

## Debug Tools

### Tool 1: Debug Page
Access: `https://www.alonedeals.com/debug_page.php`

This will show:
- Which files are loading
- Which functions exist
- Any PHP errors

### Tool 2: Test Errors
Access: `https://www.alonedeals.com/test_errors.php`

This will show if PHP error display is working.

### Tool 3: View Page Source

1. Right-click on page
2. Click "View Page Source" (not Inspect)
3. Search for `assets.alonedeals.com`
4. If you find it, those files are trying to load from CDN

## Common Issues and Fixes

### Issue 1: jQuery Not Loading

**Symptom:** Console shows `$ is not defined`

**Fix:** Check `views/includes/header-js.php`:
```php
<script src="<?php echo getHeaderJsUrl('jquery.js'); ?>"></script>
```

Make sure `$cdn_js_url = ''` is empty.

### Issue 2: main.js Not Loading

**Symptom:** Preloader never disappears

**Fix:** Check `views/includes/footer-js.php`:
```php
<script src="<?php echo getJsUrl('main.js'); ?>"></script>
```

Make sure `$cdn_js_url = ''` is empty.

### Issue 3: CSS Not Loading

**Symptom:** Page has no styling

**Fix:** Check `views/includes/header-css.php`:
```php
$cdn_css_url = ''; // Must be empty
```

### Issue 4: Fonts Not Loading

**Symptom:** Icons showing as squares

**Fix:** Fonts should be at `/assets/fonts/tabler-icons.*`

Check if files exist:
- `assets/fonts/tabler-icons.woff2`
- `assets/fonts/tabler-icons.woff`
- `assets/fonts/tabler-icons.ttf`

## Nuclear Option - Complete Reset

If nothing works, do this:

### 1. Backup Current Files
Download these files:
- `views/includes/header-css.php`
- `views/includes/header-js.php`
- `views/includes/footer-js.php`

### 2. Reset CDN Settings

Edit each file and set:
```php
$cdn_css_url = '';
$cdn_js_url = '';
```

### 3. Add Cache Busters

Add `?v=<?php echo time(); ?>` to all CSS/JS files:
```php
<link rel="stylesheet" href="<?php echo getCssUrl('styles.css'); ?>?v=<?php echo time(); ?>">
<script src="<?php echo getJsUrl('main.js'); ?>?v=<?php echo time(); ?>"></script>
```

### 4. Disable Preloader Temporarily

In `views/header.view.php`, comment out preloader:
```html
<!-- DISABLED TEMPORARILY
<div id="preloader">
...
</div>
-->
```

### 5. Clear Everything

- Clear browser cache
- Clear server cache
- Delete browser cookies
- Test in Incognito mode

## Still Not Working?

If the site still doesn't work:

1. **Check PHP error log:**
   - Look for `error_log` file in root directory
   - Or check server error logs in cPanel

2. **Contact hosting support:**
   - They can check server-side errors
   - They can verify file permissions

3. **Restore from backup:**
   - If you have a backup, restore it
   - Then make changes one at a time

## Success Checklist

- [ ] No errors in browser console
- [ ] All files loading from `www.alonedeals.com` (not CDN)
- [ ] Preloader disappears after page loads
- [ ] Footer is visible
- [ ] All styles are applied
- [ ] Icons are showing correctly
- [ ] No CORS errors
