# External Image URL Feature

## Overview
This feature allows you to use external image URLs (from any CDN or image hosting service) instead of uploading images to your server. Simply paste a URL and it will be stored and displayed directly from that URL.

## How It Works

### 1. Database Fields
Each table has TWO fields for external image support:
- `stores` → `store_image_cdn_url` (URL) + `store_use_cdn` (1/0)
- `coupons` → `coupon_image_cdn_url` (URL) + `coupon_use_cdn` (1/0)
- `campaigns` → `campaign_image_cdn_url` (URL) + `campaign_use_cdn` (1/0)
- `sliders` → `slider_image_cdn_url` (URL) + `slider_use_cdn` (1/0)
- `popups` → `popup_image_cdn_url` (URL) + `popup_use_cdn` (1/0)
- `categories` → `category_image_cdn_url` (URL) + `category_use_cdn` (1/0)

### 2. Image Priority
The system checks images in this order:
1. **External URL** - If `use_cdn = 1` AND `cdn_url` has a valid URL, use it
2. **Local Image** - If CDN is disabled or no URL, use the local uploaded image
3. **Placeholder** - If neither exists, show placeholder

### 3. Setup Instructions

#### Step 1: Run SQL Migration
Import the SQL file to add the URL fields:
```bash
File: admin/sql/add_external_image_urls.sql
```

Run this in phpMyAdmin or your MySQL client.

**Note:** If you already have `store_image_cdn_url` and `store_use_cdn` fields from a previous CDN setup, you can skip this step!

#### Step 2: Using External URLs in Admin

**For Stores:**

1. Go to Admin → Stores → Edit Store
2. In the image section, you'll see three options:
   - **Keep Current Image** - Don't change anything
   - **Upload New Image** - Upload from your computer
   - **Use Image URL** - Paste external image URL

3. Select "Use Image URL" and paste your image URL:
   ```
   https://cdn.example.com/logo.png
   https://pub-abc123.r2.dev/stores/amazon.png
   https://your-cdn.com/images/store-logo.webp
   ```

4. Click Save - **The URL is saved directly, NOT downloaded!**

#### Step 3: Update Frontend Views

In your view files, replace the old image code with the new helper function:

**Old way (local only):**
```php
<img src="<?php echo $urlPath->image($item['store_image']); ?>">
```

**New way (external URL support):**
```php
<img src="<?php echo getImageUrl($item['store_image_cdn_url'], $item['store_use_cdn'], $item['store_image']); ?>">
```

**Example for featured stores:**
```php
<!-- sections/views/featured-stores.view.php -->
<img class="store-card-logo" 
     src="<?php echo getImageUrl($item['store_image_cdn_url'], $item['store_use_cdn'], $item['store_image']); ?>" 
     alt="<?php echo echoOutput($item['store_title']); ?>">
```

### 4. Helper Function Usage

The `getImageUrl()` function takes 3 parameters:

```php
getImageUrl($cdnUrl, $useCdn, $localImage)
```

- **$cdnUrl** - The external URL field (e.g., `$item['store_image_cdn_url']`)
- **$useCdn** - Whether CDN is enabled (e.g., `$item['store_use_cdn']`)
- **$localImage** - The local image filename (e.g., `$item['store_image']`)

### 5. Benefits

✅ **No Downloads** - URLs are stored directly, not downloaded
✅ **Fast Loading** - Images load from external CDN/servers
✅ **Save Server Space** - No need to store images locally
✅ **Easy Management** - Just paste URLs, no file uploads
✅ **Flexible** - Works with any image hosting service
✅ **Backward Compatible** - Existing local images still work

### 6. Supported Image Sources

You can use images from:
- Cloudflare R2
- Amazon S3
- Google Cloud Storage
- Imgur
- Any CDN service
- Direct image URLs

### 7. Example URLs

```
https://pub-123abc.r2.dev/stores/amazon-logo.png
https://cdn.yoursite.com/images/store-logo.jpg
https://i.imgur.com/abc123.png
https://storage.googleapis.com/bucket/image.webp
https://example.com/path/to/image.jpg
```

### 8. Important Notes

⚠️ **URL Storage** - URLs are stored in database, NOT downloaded
⚠️ **HTTPS Required** - Use HTTPS URLs for security
⚠️ **Image Formats** - Supports JPG, PNG, GIF, WEBP
⚠️ **Fallback** - If external URL fails to load, browser shows broken image (consider adding onerror handler)

### 9. Troubleshooting

**Image not showing?**
1. Check if URL is valid and accessible
2. Verify `use_cdn` field is set to 1
3. Check if URL uses HTTPS
4. Open URL directly in browser to test
5. Check browser console for errors

**Want to switch back to local?**
1. Edit the item in admin
2. Select "Upload New Image"
3. Upload a new file
4. The system will use local image instead

### 10. Migration from Old System

If you have existing local images:
1. They will continue to work automatically
2. No changes needed
3. You can gradually switch to external URLs
4. Or keep using local images - both work!

---

## Technical Details

### Database Schema
```sql
-- These fields should already exist if you ran the CDN setup before
ALTER TABLE stores ADD COLUMN store_image_cdn_url VARCHAR(500) DEFAULT '';
ALTER TABLE stores ADD COLUMN store_use_cdn TINYINT(1) DEFAULT 0;
-- Repeat for coupons, campaigns, sliders, popups, categories
```

### PHP Helper Function
```php
function getImageUrl($cdnUrl, $useCdn, $localImage) {
    // If CDN is enabled and URL is valid, use it
    if ($useCdn == 1 && !empty($cdnUrl) && filter_var($cdnUrl, FILTER_VALIDATE_URL)) {
        return $cdnUrl;
    }
    // Otherwise use local image
    if (!empty($localImage)) {
        return SITE_URL . '/images/' . $localImage;
    }
    // Fallback to placeholder
    return SITE_URL . '/images/placeholder.png';
}
```

### Controller Logic (Stores)
```php
// When user selects "Use Image URL"
if ($image_type == 'url' && !empty($_POST['image_url'])) {
    $image_url = trim($_POST['image_url']);
    
    // Validate URL
    if (filter_var($image_url, FILTER_VALIDATE_URL)) {
        $store_image_cdn_url = $image_url;  // Save URL
        $store_use_cdn = 1;                  // Enable CDN
        // No download happens!
    }
}
```

---

**That's it!** Simple URL-based image system with no downloads, no complex CDN integration, just pure URL storage and display.
