# Performance Optimization Guide

## Current Issues

You're seeing these browser warnings:
- `[Violation] 'requestAnimationFrame' handler took 106ms`
- `[Violation] Forced reflow while executing JavaScript took 54ms`

These are **performance warnings**, not errors. Your site still works, but it's slower than optimal.

## What Causes These Warnings?

1. **UIKit animations** - Heavy CSS/JS animations
2. **Multiple Swiper sliders** - All initializing at once
3. **Large images** - Not optimized or lazy-loaded
4. **Too many DOM operations** - JavaScript manipulating HTML too much

## Quick Fixes (Easy)

### 1. Lazy Load Images
Add `loading="lazy"` to images that are below the fold:

```php
<img src="<?php echo $image_url; ?>" loading="lazy" alt="...">
```

### 2. Defer Non-Critical JavaScript
In `views/includes/footer-js.php`, add `defer` to scripts:

```html
<script src="..." defer></script>
```

### 3. Reduce Swiper Autoplay Speed
Slower autoplay = less CPU usage. In your Swiper configs, increase delay:

```javascript
autoplay: {
    delay: 5000, // Change from 3000 to 5000
    disableOnInteraction: false,
}
```

### 4. Disable Loop on Sliders with Few Items
If you have less than 5 items, disable loop:

```javascript
loop: false, // Instead of loop: true
```

## Medium Fixes (Moderate Effort)

### 1. Use Intersection Observer for Sliders
Only initialize sliders when they're visible:

```javascript
const observer = new IntersectionObserver((entries) => {
    entries.forEach(entry => {
        if (entry.isIntersecting) {
            // Initialize Swiper here
            observer.unobserve(entry.target);
        }
    });
});

observer.observe(document.querySelector('.campaigns-swiper'));
```

### 2. Optimize Images
- Use WebP format instead of PNG/JPG
- Compress images (use TinyPNG or similar)
- Resize images to actual display size (don't use 2000px images for 200px display)

### 3. Reduce UIKit Usage
UIKit is heavy. Consider replacing some UIKit components with vanilla CSS/JS.

## Advanced Fixes (More Effort)

### 1. Code Splitting
Load JavaScript only when needed, not all at once.

### 2. Use CDN for Static Assets
Serve images, CSS, JS from a CDN (you're already working on this!)

### 3. Enable Browser Caching
Add to `.htaccess`:

```apache
<IfModule mod_expires.c>
    ExpiresActive On
    ExpiresByType image/jpg "access plus 1 year"
    ExpiresByType image/jpeg "access plus 1 year"
    ExpiresByType image/png "access plus 1 year"
    ExpiresByType image/webp "access plus 1 year"
    ExpiresByType text/css "access plus 1 month"
    ExpiresByType application/javascript "access plus 1 month"
</IfModule>
```

### 4. Minify CSS and JavaScript
Use tools to compress your CSS/JS files.

## Should You Worry?

**No, not immediately.** These are warnings, not errors. Your site works fine. But if you want better performance:

1. Start with the "Quick Fixes" above
2. Test on slow devices/connections
3. Use Chrome DevTools Performance tab to identify bottlenecks
4. Optimize the slowest parts first

## Testing Performance

1. Open Chrome DevTools (F12)
2. Go to "Lighthouse" tab
3. Click "Generate report"
4. See your performance score and recommendations

## Priority Actions

**Do these first:**
1. ✅ Add `loading="lazy"` to images below the fold
2. ✅ Increase Swiper autoplay delay from 3000ms to 5000ms
3. ✅ Disable loop on sliders with < 5 items
4. ✅ Compress large images

**Do these later:**
- Implement Intersection Observer for sliders
- Convert images to WebP format
- Enable browser caching
- Minify CSS/JS files

## Current Status

Your site is functional. These warnings are just the browser saying "hey, this could be faster." Most users won't notice unless they're on very slow devices.

Focus on your CDN implementation first (which you're already doing), then come back to these performance optimizations.
