<?php
/**
 * Apply Render-Blocking Fix
 * 
 * This script automatically updates header.view.php and footer.view.php
 * to use the optimized CSS/JS loading files.
 * 
 * Usage: php apply_render_blocking_fix.php
 */

echo "=== Render-Blocking Resources Fix ===\n\n";

// Check if optimized files exist
$required_files = [
    'views/includes/header-css-optimized.php',
    'views/includes/header-js-optimized.php',
    'views/includes/footer-js-optimized.php'
];

$missing_files = [];
foreach ($required_files as $file) {
    if (!file_exists($file)) {
        $missing_files[] = $file;
    }
}

if (!empty($missing_files)) {
    echo "ERROR: Missing required files:\n";
    foreach ($missing_files as $file) {
        echo "  - $file\n";
    }
    exit(1);
}

echo "✓ All optimized files found\n\n";

// Backup original files
$backup_suffix = '_backup_' . date('Y-m-d_H-i-s');

$files_to_backup = [
    'views/header.view.php',
    'views/footer.view.php',
    'views/includes/header-css.php',
    'views/includes/header-js.php',
    'views/includes/footer-js.php'
];

echo "Creating backups...\n";
foreach ($files_to_backup as $file) {
    if (file_exists($file)) {
        $backup_file = $file . $backup_suffix;
        if (copy($file, $backup_file)) {
            echo "  ✓ Backed up: $file -> $backup_file\n";
        } else {
            echo "  ✗ Failed to backup: $file\n";
            exit(1);
        }
    }
}

echo "\n";

// Update header.view.php
echo "Updating views/header.view.php...\n";
$header_content = file_get_contents('views/header.view.php');

// Replace header-css.php with header-css-optimized.php
$header_content = str_replace(
    "include __DIR__ . '/includes/header-css.php';",
    "include __DIR__ . '/includes/header-css-optimized.php';",
    $header_content
);

// Replace header-js.php with header-js-optimized.php
$header_content = str_replace(
    "include __DIR__ . '/includes/header-js.php';",
    "include __DIR__ . '/includes/header-js-optimized.php';",
    $header_content
);

if (file_put_contents('views/header.view.php', $header_content)) {
    echo "  ✓ Updated header.view.php\n";
} else {
    echo "  ✗ Failed to update header.view.php\n";
    exit(1);
}

// Update footer.view.php
echo "Updating views/footer.view.php...\n";
$footer_content = file_get_contents('views/footer.view.php');

// Replace footer-js.php with footer-js-optimized.php
$footer_content = str_replace(
    "include __DIR__ . '/includes/footer-js.php';",
    "include __DIR__ . '/includes/footer-js-optimized.php';",
    $footer_content
);

if (file_put_contents('views/footer.view.php', $footer_content)) {
    echo "  ✓ Updated footer.view.php\n";
} else {
    echo "  ✗ Failed to update footer.view.php\n";
    exit(1);
}

echo "\n";
echo "=== SUCCESS ===\n";
echo "Render-blocking fix has been applied!\n\n";
echo "Next steps:\n";
echo "1. Test your site on localhost\n";
echo "2. Check all pages load correctly\n";
echo "3. Verify all functionality works (sliders, modals, forms)\n";
echo "4. Upload to production server\n";
echo "5. Clear Cloudflare cache\n";
echo "6. Test with PageSpeed Insights\n\n";
echo "To rollback, restore these backup files:\n";
foreach ($files_to_backup as $file) {
    if (file_exists($file . $backup_suffix)) {
        echo "  - $file$backup_suffix\n";
    }
}
echo "\n";
?>
